sir_sample <- function(vec, n, weight_func, proposed_func) {
  samples <- matrix(nrow = n, ncol = length(vec))
  weights <- rep(NA, n)
  indices <- sample.int(n, size = n, replace = TRUE)

  for (i in 1:n) {
    proposed <- proposed_func(vec[indices[i], ])
    weights[i] <- weight_func(proposed)
    samples[i, ] <- proposed
  }
  weights[is.na(weights) | weights < 0] <- 0
  if(sum(weights) == 0){
    weights <- rep(1/length(weights), length(weights))
  } else {
    weights <- weights / sum(weights)
  }
  resampled_indices <- sample.int(n, size = n, replace = TRUE, prob = weights)
  return(samples[resampled_indices, ])
}

#' Title 11.3(without_cia)
#'
#' @param init_pi Initial value of prevalence
#' @param init_sens1 Initial value of sensitivity
#' @param init_spec1 Initial value of specificity
#' @param init_sens2 Initial value of sensitivity
#' @param init_spec2 Initial value of specificity
#' @param init_covc12 The initial value of the covariance parameter
#' @param init_covs12 The initial value of the covariance parameter
#' @param data Observations, a four-dimensional vector
#' @param n_iter Number of iterations
#' @param burn_in burn_in period
#' @param alpha.pi Prior information on prevalence
#' @param beta.pi Prior information on prevalence
#' @param alpha.sens1 Prior information on sensitivity
#' @param beta.sens1 Prior information on sensitivity
#' @param alpha.spec1 Prior information on specificity
#' @param beta.spec1 Prior information on specificity
#' @param alpha.sens2 Prior information on sensitivity
#' @param beta.sens2 Prior information on sensitivity
#' @param alpha.spec2 Prior information on specificity
#' @param beta.spec2 Prior information on specificity
#' @param alpha.covs12 Prior information on covariance parameters
#' @param beta.covs12 Prior information on covariance parameters
#' @param alpha.covc12 Prior information on covariance parameters
#' @param beta.covc12 Prior information on covariance parameters
#'
#' @return Means and 95% confidence intervals for prevalence, sensitivity and specificity
#' @export
#'
#' @examples gibbs_sampler(init_pi=0.5, init_sens1=0.95, init_spec1=0.97, init_sens2=0.9, init_spec2=0.95, init_covc12=0.5, init_covs12=0.5, data=c(38,2,87,35), n_iter=2000, burn_in=500, alpha.pi = 1, beta.pi = 1, alpha.sens1 = 4.44,beta.sens1 = 13.31, alpha.spec1 = 71.25, beta.spec1 = 3.75,alpha.sens2 = 21.96, beta.sens2 = 5.49, alpha.spec2 = 4.1,beta.spec2 = 1.76, alpha.covs12 = 1, beta.covs12 = 1, alpha.covc12 = 1,beta.covc12 = 1)
gibbs_sampler <- function(init_pi, init_sens1, init_spec1, init_sens2, init_spec2,
                          init_covc12, init_covs12, data, n_iter, burn_in, alpha.pi,
                          beta.pi, alpha.sens1, beta.sens1,
                          alpha.spec1, beta.spec1, alpha.sens2,
                          beta.sens2, alpha.spec2, beta.spec2,
                          alpha.covs12, beta.covs12, alpha.covc12,
                          beta.covc12){
  set.seed(1234)
  n <- sum(data)
  n_param <- 7
  trace <- matrix(0, n_iter-burn_in+1, n_param)
  colnames(trace) <- c("pi", "sens1", "spec1", "sens2", "spec2", "covc12", "covs12")
  index <- 1

  pi <- init_pi
  sens1 <- init_sens1
  spec1 <- init_spec1
  sens2 <- init_sens2
  spec2 <- init_spec2
  uc <- min(spec1, spec2)-spec1*spec2
  us <- min(sens1, sens2)-sens1*sens2
  covc12 <- init_covc12
  covs12 <- init_covs12

  for (i in 1:n_iter) {

    Y11 <- rbinom(data[1], 1, pi*(sens1*sens2+covs12)/(pi*(sens1*sens2+covs12) +
                                                         (1-pi)*((1-spec1)*(1-spec2)+covc12) ))
    Y10 <- rbinom(data[2], 1, pi*(sens1*(1-sens2)-covs12)/(pi*(sens1*(1- sens2)-covs12) +
                                                             (1-pi)*((1-spec1)*spec2-covc12)))
    Y01 <- rbinom(data[3], 1, pi*((1-sens1)*sens2-covs12)/(pi*((1-sens1)*sens2-covs12) +
                                                             (1-pi)*(spec1*(1-spec2)-covc12)))
    Y00 <- rbinom(data[4], 1, pi*((1-sens1)*(1-sens2)+covs12)/(pi*((1-sens1)*(1-sens2)+covs12)+
                                                                 (1-pi)*(spec1*spec2+covc12)))
    alpha.pi.updated <- sum(Y11) + sum(Y10) + sum(Y01) + sum(Y00) + alpha.pi
    beta.pi.updated <- sum(data) - (sum(Y11) + sum(Y10) + sum(Y01) + sum(Y00)) + beta.pi
    pi<-rbeta(1, alpha.pi.updated, beta.pi.updated)

    alpha_sens1_updated <- alpha.sens1
    beta_sens1_updated  <- beta.sens1

    weight_func <- function(proposed) {
      candidate_sens1 <- proposed[1]
      candidate_covs12 <- proposed[2]
      (candidate_sens1^(alpha_sens1_updated - 1) * (1 - candidate_sens1)^(beta_sens1_updated - 1) *
          (candidate_sens1 * sens2 + candidate_covs12)^sum(Y11)*
          (candidate_sens1*(1 - sens2) - candidate_covs12)^sum(Y10)*
          ((1 - candidate_sens1) * sens2 - candidate_covs12)^sum(Y01)*
          ((1 - candidate_sens1) * (1 - sens2) + candidate_covs12)^sum(Y00)*
          candidate_covs12^(alpha.covs12 - 1) * (us - candidate_covs12)^(beta.covs12 - 1))
    }

    proposed_func <- function(curr) {
      candidate_sens1 <- runif(1)
      candidate_covs12 <- runif(1, min = 0, max = us)
      c(candidate_sens1, candidate_covs12)
    }
    samples <- sir_sample(c(sens1, covs12), n = 1000, weight_func, proposed_func)
    sens1 <- sample(samples[, 1], size = 1)

    weight_func <- function(proposed) {
      candidate_sens2 <- proposed[1]
      candidate_covs12 <- proposed[2]
      (candidate_sens2^(alpha.sens2-1) * (1 - candidate_sens2)^(beta.sens2-1) *
          (sens1*candidate_sens2+candidate_covs12)^sum(Y11)*
          (sens1*(1-candidate_sens2)-candidate_covs12)^sum(Y10)*
          ((1 - sens1) * candidate_sens2 - candidate_covs12)^sum(Y01)*
          ((1 - sens1)*(1 - candidate_sens2) + candidate_covs12)^sum(Y00)*
          candidate_covs12^(alpha.covs12 - 1) * (us - candidate_covs12)^(beta.covs12 - 1))
    }

    proposed_func <- function(curr) {
      candidate_sens2 <- runif(1)
      candidate_covs12 <- runif(1, min = 0, max = us)
      c(candidate_sens2, candidate_covs12)
    }
    samples <- sir_sample(c(sens2, covs12), n = 1000, weight_func, proposed_func)
    sens2<- sample(samples[, 1], size = 1)

    us <- min(sens1, sens2) - sens1*sens2

    weight_func <- function(proposed) {
      candidate_spec1 <- proposed[1]
      candidate_covc12 <- proposed[2]
      ((1 - candidate_spec1)*(1-spec2)+candidate_covc12)^(data[1] - sum(Y11)) *
        ((1 - candidate_spec1) * spec2 - candidate_covc12)^(data[2] - sum(Y10)) *
        (candidate_spec1 * (1 - spec2) - candidate_covc12)^(data[3] - sum(Y01)) *
        (spec1 * spec2 + candidate_covc12)^(data[4] - sum(Y00)) *
        candidate_spec1^(alpha.spec1 - 1) * (1 - candidate_spec1)^(beta.spec1 - 1) *
        candidate_covc12^(alpha.covc12-1)*(uc - candidate_covc12)^(beta.covc12 - 1)
    }

    proposed_func <- function(curr) {
      candidate_spec1 <- runif(1)
      candidate_covc12 <- runif(1, min = 0, max = uc)
      c(candidate_spec1, candidate_covc12)
    }

    samples <- sir_sample(c(spec1, covc12), n = 1000, weight_func, proposed_func)
    spec1 <- sample(samples[, 1], size = 1)

    weight_func <- function(proposed) {
      candidate_spec2 <- proposed[1]
      candidate_covc12 <- proposed[2]
      ((1 - spec1)*(1-candidate_spec2)+candidate_covc12)^(data[1] - sum(Y11)) *
        ((1 - spec1) * candidate_spec2 - candidate_covc12)^(data[2] - sum(Y10)) *
        (spec1 * (1 - candidate_spec2) - candidate_covc12)^(data[3] - sum(Y01)) *
        (spec1 * candidate_spec2 + candidate_covc12)^(data[4] - sum(Y00)) *
        candidate_spec2^(alpha.spec2 - 1) * (1 - candidate_spec2)^(beta.spec2 - 1) *
        candidate_covc12^(alpha.covc12-1)*(uc - candidate_covc12)^(beta.covc12 - 1)
    }

    proposed_func <- function(curr) {
      candidate_spec2 <-runif(1)
      candidate_covc12 <- runif(1, min = 0, max = uc)
      c(candidate_spec2, candidate_covc12)
    }
    samples <- sir_sample(c(spec2, covc12), n=1000, weight_func, proposed_func)
    spec2 <- sample(samples[, 1], size = 1)

    uc <- min(spec1, spec2) - spec1*spec2

    weight_func <- function(proposed) {
      candidate_covs12 <- proposed[1]
      ((sens1 * sens2 + candidate_covs12)^sum(Y11) *
          (sens1 * (1 - sens2) - candidate_covs12)^sum(Y10) *
          ((1 - sens1) * sens2 - candidate_covs12)^sum(Y01) *
          ((1 - sens1) * (1 - sens2) + candidate_covs12)^sum(Y00) *
          candidate_covs12^(alpha.covs12 - 1) * (us - candidate_covs12)^(beta.covs12 - 1))
    }

    proposed_func <- function(curr) runif(1, min = 0, max = us)

    samples <- sir_sample(covs12, n = 1000, weight_func, proposed_func)
    covs12 <- sample(samples, size = 1)

    weight_func <- function(proposed) {
      candidate_covc12 <- proposed[1]
      (((1 - spec1)*(1- spec2)+candidate_covc12)^(data[1] - sum(Y11)) *
          ((1 - spec1) * spec2 - candidate_covc12)^(data[2] - sum(Y10)) *
          (spec1 * (1 - spec2) - candidate_covc12)^(data[3] - sum(Y01)) *
          (spec1 * spec2 + candidate_covc12)^(data[4] - sum(Y00)) *
          candidate_covc12^(alpha.covc12 - 1) * (uc - candidate_covc12)^(beta.covc12 - 1))
    }

    proposed_func <- function(curr) runif(1, min = 0, max = uc)

    samples <- sir_sample(covc12, n = 1000, weight_func, proposed_func)
    covc12 <- sample(samples, size = 1)

    if(i > burn_in){
      trace[index, ] <- c(pi, sens1, spec1, sens2, spec2, covs12, covc12)
      index <- index+1
    }
  }
  result <- list()
  for(j in 1:n_param){
    result[[colnames(trace)[j]]] <- c(median(trace[, j]),
                                      quantile(trace[, j], c(0.025, 0.975)))
  }
  return(result)
}
